using System;
using System.IO;
using SautinSoft;
using SautinSoft.Pdf;
using SautinSoft.Pdf.Content;

namespace Sample
{
    class Sample
    {
        /// <summary>
        /// Add shapes to PDF files.
        /// </summary>
        /// <remarks>
        /// Details: https://sautinsoft.com/products/pdf/help/net/developer-guide/add-shapes-to-pdf.php
        /// </remarks>
        static void Main(string[] args)
        {
            // Before starting this example, please get a free trial key:
            // https://sautinsoft.com/start-for-free/

            // Apply the key here:
            // PdfDocument.SetLicense("...");

            using (var document = new PdfDocument())
            {
                // Add a page.
                var page = document.Pages.Add();

                // NOTE: In PDF, location (0, 0) is at the bottom-left corner of the page
                // and the positive y axis extends vertically upward.
                var pageBounds = page.CropBox;

                // Add a thick red line at the top of the page.
                var line = page.Content.Elements.AddPath();
                line.BeginSubpath(new PdfPoint(100, pageBounds.Top - 100)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, pageBounds.Top - 200));
                var lineFormat = line.Format;
                lineFormat.Stroke.IsApplied = true;
                lineFormat.Stroke.Width = 5;
                lineFormat.Stroke.Color = PdfColor.FromRgb(1, 0, 0);

                // Add a filled and stroked rectangle in the middle of the page.
                var rectangle = page.Content.Elements.AddPath();
                // NOTE: The start point of the rectangle is the bottom left corner of the rectangle.
                rectangle.AddRectangle(new PdfPoint(100, pageBounds.Top - 400),
                    new PdfSize(pageBounds.Width - 200, 100));
                var rectangleFormat = rectangle.Format;
                rectangleFormat.Fill.IsApplied = true;
                rectangleFormat.Fill.Color = PdfColor.FromRgb(0, 1, 0);
                rectangleFormat.Stroke.IsApplied = true;
                rectangleFormat.Stroke.Width = 10;
                rectangleFormat.Stroke.Color = PdfColor.FromRgb(0, 0, 1);

                // Add a more complex semi-transparent filled and stroked path at the bottom of the page.
                var shape = page.Content.Elements.AddPath();
                shape.BeginSubpath(new PdfPoint(100, 100)).
                    BezierTo(new PdfPoint(100 + pageBounds.Width / 4, 200),
                        new PdfPoint(pageBounds.Right - 100 - pageBounds.Width / 4, 0),
                        new PdfPoint(pageBounds.Right - 100, 100)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, 300)).
                    BezierTo(new PdfPoint(pageBounds.Right - 100 - pageBounds.Width / 4, 200),
                        new PdfPoint(100 + pageBounds.Width / 4, 400),
                        new PdfPoint(100, 300)).
                    CloseSubpath();
                var shapeFormat = shape.Format;
                shapeFormat.Fill.IsApplied = true;
                shapeFormat.Fill.Color = PdfColor.FromRgb(0, 1, 0);
                shapeFormat.Fill.Opacity = 0.5;
                shapeFormat.Stroke.IsApplied = true;
                shapeFormat.Stroke.Width = 4;
                shapeFormat.Stroke.Color = PdfColor.FromRgb(0, 0, 1);
                shapeFormat.Stroke.Opacity = 0.5;
                shapeFormat.Stroke.DashPattern = PdfLineDashPatterns.DashDot;

                // Add a grid to visualize the bounds of each drawn shape.
                var grid = page.Content.Elements.AddPath();
                grid.AddRectangle(new PdfPoint(100, 100),
                    new PdfSize(pageBounds.Width - 200, pageBounds.Height - 200));
                grid.BeginSubpath(new PdfPoint(100, pageBounds.Top - 200)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, pageBounds.Top - 200)).
                    BeginSubpath(new PdfPoint(100, pageBounds.Top - 300)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, pageBounds.Top - 300)).
                    BeginSubpath(new PdfPoint(100, pageBounds.Top - 400)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, pageBounds.Top - 400)).
                    BeginSubpath(new PdfPoint(100, 300)).
                    LineTo(new PdfPoint(pageBounds.Right - 100, 300));
                grid.Format.Stroke.IsApplied = true;
                // A line width of 0 denotes the thinnest line that can be rendered at device resolution: 1 device pixel wide.
                grid.Format.Stroke.Width = 0;

                document.Save("Paths.pdf");
            }

            System.Diagnostics.Process.Start(new System.Diagnostics.ProcessStartInfo("Paths.pdf") { UseShellExecute = true });
        }
    }
}
